<?php
require_once 'auth_check.php';

$pageTitle = "Kelola Tags";

// Process actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        $tagId = isset($_POST['tag_id']) ? (int)$_POST['tag_id'] : 0;
        
        try {
            switch ($action) {
                case 'add':
                    $name = trim($_POST['name']);
                    
                    if (empty($name)) {
                        $errorMessage = "Nama tag tidak boleh kosong";
                        break;
                    }
                    
                    $stmt = $db->prepare("INSERT INTO tags (name) VALUES (?)");
                    $stmt->execute([$name]);
                    $successMessage = "Tag berhasil ditambahkan";
                    break;
                    
                case 'edit':
                    $name = trim($_POST['name']);
                    
                    if (empty($name)) {
                        $errorMessage = "Nama tag tidak boleh kosong";
                        break;
                    }
                    
                    $stmt = $db->prepare("UPDATE tags SET name = ? WHERE id = ?");
                    $stmt->execute([$name, $tagId]);
                    $successMessage = "Tag berhasil diperbarui";
                    break;
                    
                case 'delete':
                    // Check if tag is used
                    $stmt = $db->prepare("SELECT COUNT(*) FROM file_tags WHERE tag_id = ?");
                    $stmt->execute([$tagId]);
                    $usedCount = $stmt->fetchColumn();
                    
                    if ($usedCount > 0) {
                        $errorMessage = "Tag tidak dapat dihapus karena masih digunakan oleh $usedCount file";
                        break;
                    }
                    
                    $stmt = $db->prepare("DELETE FROM tags WHERE id = ?");
                    $stmt->execute([$tagId]);
                    $successMessage = "Tag berhasil dihapus";
                    break;
            }
        } catch (PDOException $e) {
            if ($e->errorInfo[1] == 1062) {
                $errorMessage = "Nama tag sudah ada";
            } else {
                $errorMessage = "Database error: " . $e->getMessage();
            }
        }
    }
}

// Get all tags with file counts
$tags = $db->query("
    SELECT t.*, COUNT(ft.file_id) as file_count 
    FROM tags t
    LEFT JOIN file_tags ft ON t.id = ft.tag_id
    GROUP BY t.id
    ORDER BY t.name
")->fetchAll();

include 'include/header.php';
?>

<div class="container-fluid px-4">
    <h1 class="mt-4">Kelola Tags</h1>
    <ol class="breadcrumb mb-4">
        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
        <li class="breadcrumb-item active">Tags</li>
    </ol>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $successMessage ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $errorMessage ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-4">
            <div class="card mb-4">
                <div class="card-header">
                    <i class="bi bi-plus-circle me-1"></i> Tambah Tag Baru
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="action" value="add">
                        
                        <div class="mb-3">
                            <label for="name" class="form-label">Nama Tag*</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                            <div class="form-text">Gunakan huruf kecil dan tanpa spasi (contoh: lua, python, script)</div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="bi bi-save me-1"></i> Simpan Tag
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-8">
            <div class="card mb-4">
                <div class="card-header">
                    <i class="bi bi-tags me-1"></i> Daftar Tags
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th width="50">ID</th>
                                    <th>Nama</th>
                                    <th>Jumlah File</th>
                                    <th width="120">Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($tags)): ?>
                                    <tr>
                                        <td colspan="4" class="text-center">Belum ada tag</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($tags as $tag): ?>
                                        <tr>
                                            <td><?= $tag['id'] ?></td>
                                            <td><?= htmlspecialchars($tag['name']) ?></td>
                                            <td><?= $tag['file_count'] ?></td>
                                            <td>
                                                <div class="d-flex gap-1">
                                                    <button type="button" class="btn btn-sm btn-warning edit-tag" 
                                                            data-id="<?= $tag['id'] ?>"
                                                            data-name="<?= htmlspecialchars($tag['name']) ?>">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-sm btn-danger delete-tag" 
                                                            data-id="<?= $tag['id'] ?>"
                                                            data-name="<?= htmlspecialchars($tag['name']) ?>">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Edit Tag -->
<div class="modal fade" id="editTagModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Tag</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="tag_id" id="editTagId" value="">
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="editTagName" class="form-label">Nama Tag*</label>
                        <input type="text" class="form-control" id="editTagName" name="name" required>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-primary">Simpan Perubahan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Hapus Tag -->
<div class="modal fade" id="deleteTagModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Konfirmasi Hapus</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="tag_id" id="deleteTagId" value="">
                
                <div class="modal-body">
                    <p>Apakah Anda yakin ingin menghapus tag <strong id="deleteTagName"></strong>?</p>
                    <p class="text-danger">Tindakan ini tidak dapat dibatalkan.</p>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-danger">Hapus</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Edit tag modal
    const editButtons = document.querySelectorAll('.edit-tag');
    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            document.getElementById('editTagId').value = this.getAttribute('data-id');
            document.getElementById('editTagName').value = this.getAttribute('data-name');
            
            const modal = new bootstrap.Modal(document.getElementById('editTagModal'));
            modal.show();
        });
    });
    
    // Delete tag modal
    const deleteButtons = document.querySelectorAll('.delete-tag');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
            document.getElementById('deleteTagId').value = this.getAttribute('data-id');
            document.getElementById('deleteTagName').textContent = this.getAttribute('data-name');
            
            const modal = new bootstrap.Modal(document.getElementById('deleteTagModal'));
            modal.show();
        });
    });
});
</script>

<?php include 'include/footer.php'; ?>