<?php
/**
 * Admin Panel - Kelola Member
 * File ini untuk manajemen member bot
 */

// Include auth check
require_once 'auth_check.php';

// Process actions
if (isset($_POST['action'])) {
    $action = $_POST['action'];
    $memberId = isset($_POST['member_id']) ? (int)$_POST['member_id'] : 0;
    
    try {
        switch ($action) {
            case 'block':
                $stmt = $db->prepare("UPDATE users SET is_blocked = 1 WHERE id = ?");
                $stmt->execute([$memberId]);
                $successMessage = "Member berhasil diblokir";
                break;
                
            case 'unblock':
                $stmt = $db->prepare("UPDATE users SET is_blocked = 0 WHERE id = ?");
                $stmt->execute([$memberId]);
                $successMessage = "Member berhasil diunblokir";
                break;
            
            case 'delete':
                // Check if user has active keys first
                $stmt = $db->prepare("SELECT COUNT(*) FROM access_keys WHERE user_id = ? AND is_used = 1");
                $stmt->execute([$memberId]);
                if ($stmt->fetchColumn() > 0) {
                    $errorMessage = "Tidak dapat menghapus member dengan key aktif";
                } else {
                    $stmt = $db->prepare("DELETE FROM users WHERE id = ?");
                    $stmt->execute([$memberId]);
                    $successMessage = "Member berhasil dihapus";
                }
                break;
        }
    } catch (PDOException $e) {
        $errorMessage = "Database error: " . $e->getMessage();
    }
}

// Filter settings
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
$search = isset($_GET['search']) ? $_GET['search'] : '';

// Build query based on filter
$query = "SELECT u.*, 
          (SELECT COUNT(*) FROM access_keys WHERE user_id = u.id AND is_used = 1) as has_premium
          FROM users u WHERE 1=1";

if ($filter === 'premium') {
    $query .= " AND (SELECT COUNT(*) FROM access_keys WHERE user_id = u.id AND is_used = 1) > 0";
} elseif ($filter === 'blocked') {
    $query .= " AND is_blocked = 1";
} elseif ($filter === 'active') {
    $query .= " AND is_blocked = 0";
}

if (!empty($search)) {
    $query .= " AND (username LIKE :search OR first_name LIKE :search OR last_name LIKE :search OR telegram_id LIKE :search)";
}

$query .= " ORDER BY created_at DESC";

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Get total count for pagination
$countQuery = str_replace("SELECT u.*, (SELECT COUNT(*) FROM access_keys WHERE user_id = u.id AND is_used = 1) as has_premium", "SELECT COUNT(*)", $query);
$stmt = $db->prepare($countQuery);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->execute();
$totalMembers = $stmt->fetchColumn();
$totalPages = ceil($totalMembers / $perPage);

// Get members with limit
$query .= " LIMIT :offset, :perPage";
$stmt = $db->prepare($query);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
$stmt->execute();
$members = $stmt->fetchAll();

// Get member statistics
$stmt = $db->query("SELECT COUNT(*) as total FROM users");
$totalAllMembers = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM users WHERE is_blocked = 1");
$totalBlockedMembers = $stmt->fetch()['total'];

$stmt = $db->query("
    SELECT COUNT(DISTINCT user_id) as total 
    FROM access_keys 
    WHERE is_used = 1 AND user_id IS NOT NULL
");
$totalPremiumMembers = $stmt->fetch()['total'];

// Page title
$pageTitle = "Kelola Member";
?>

<?php include 'include/header.php'; ?>

<div class="container-fluid px-4">
    <h1 class="mt-4">Kelola Member</h1>
    <ol class="breadcrumb mb-4">
        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
        <li class="breadcrumb-item active">Kelola Member</li>
    </ol>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success" role="alert">
            <?php echo $successMessage; ?>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger" role="alert">
            <?php echo $errorMessage; ?>
        </div>
    <?php endif; ?>
    
    <!-- Statistics cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6">
            <div class="card bg-primary text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalAllMembers; ?></h4>
                    <div>Total Member</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6">
            <div class="card bg-success text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalAllMembers - $totalBlockedMembers; ?></h4>
                    <div>Member Aktif</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6">
            <div class="card bg-warning text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalPremiumMembers; ?></h4>
                    <div>Member Premium</div>
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6">
            <div class="card bg-danger text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalBlockedMembers; ?></h4>
                    <div>Member Diblokir</div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Member list -->
    <div class="card mb-4">
        <div class="card-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <i class="bi bi-people me-1"></i>
                    Daftar Member
                </div>
                <div>
                    <a href="?filter=all" class="btn btn-sm <?php echo $filter === 'all' ? 'btn-primary' : 'btn-outline-primary'; ?>">Semua</a>
                    <a href="?filter=active" class="btn btn-sm <?php echo $filter === 'active' ? 'btn-success' : 'btn-outline-success'; ?>">Aktif</a>
                    <a href="?filter=premium" class="btn btn-sm <?php echo $filter === 'premium' ? 'btn-warning' : 'btn-outline-warning'; ?>">Premium</a>
                    <a href="?filter=blocked" class="btn btn-sm <?php echo $filter === 'blocked' ? 'btn-danger' : 'btn-outline-danger'; ?>">Diblokir</a>
                </div>
            </div>
        </div>
        <div class="card-body">
            <!-- Search form -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <form action="" method="GET" class="d-flex">
                        <input type="hidden" name="filter" value="<?php echo $filter; ?>">
                        <input type="text" name="search" id="searchInput" class="form-control me-2" placeholder="Cari member..." value="<?php echo htmlspecialchars($search); ?>">
                        <button type="submit" class="btn btn-primary">Cari</button>
                    </form>
                </div>
            </div>
            
            <!-- Members table -->
            <div class="table-responsive">
                <table class="table table-striped table-hover" id="dataTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Username</th>
                            <th>Nama</th>
                            <th>Telegram ID</th>
                            <th>Status</th>
                            <th>Terdaftar</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($members)): ?>
                            <tr>
                                <td colspan="7" class="text-center">Tidak ada data member</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($members as $member): ?>
                                <tr>
                                    <td><?php echo $member['id']; ?></td>
                                    <td>
                                        <?php if (!empty($member['username'])): ?>
                                            @<?php echo htmlspecialchars($member['username']); ?>
                                        <?php else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                            $name = trim($member['first_name'] . ' ' . $member['last_name']);
                                            echo !empty($name) ? htmlspecialchars($name) : '<span class="text-muted">-</span>';
                                        ?>
                                    </td>
                                    <td><?php echo $member['telegram_id']; ?></td>
                                    <td>
                                        <?php if ($member['is_blocked']): ?>
                                            <span class="badge bg-danger">Diblokir</span>
                                        <?php else: ?>
                                            <span class="badge bg-success">Aktif</span>
                                        <?php endif; ?>
                                        
                                        <?php if ($member['has_premium'] > 0): ?>
                                            <span class="badge bg-warning">Premium</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($member['created_at'])); ?></td>
                                    <td>
                                        <form action="" method="POST" class="d-inline">
                                            <input type="hidden" name="member_id" value="<?php echo $member['id']; ?>">
                                            
                                            <?php if ($member['is_blocked']): ?>
                                                <button type="submit" name="action" value="unblock" class="btn btn-sm btn-success">
                                                    <i class="bi bi-unlock"></i> Unblock
                                                </button>
                                            <?php else: ?>
                                                <button type="submit" name="action" value="block" class="btn btn-sm btn-warning">
                                                    <i class="bi bi-lock"></i> Block
                                                </button>
                                            <?php endif; ?>
                                            
                                            <button type="submit" name="action" value="delete" class="btn btn-sm btn-danger btn-delete">
                                                <i class="bi bi-trash"></i> Hapus
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?php echo ($page <= 1) ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $page-1; ?>&search=<?php echo urlencode($search); ?>">Previous</a>
                        </li>
                        
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? 'active' : ''; ?>">
                                <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                        
                        <li class="page-item <?php echo ($page >= $totalPages) ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $page+1; ?>&search=<?php echo urlencode($search); ?>">Next</a>
                        </li>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'include/footer.php'; ?>
