<?php
/**
 * Admin Panel - Kelola File
 * File ini untuk manajemen file yang dibagikan melalui bot
 */

// Include auth check
require_once 'auth_check.php';

// Directory for file uploads
$uploadDir = __DIR__ . '/../public/files/' . date('Y-m-d');
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Process actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        $fileId = isset($_POST['file_id']) ? (int)$_POST['file_id'] : 0;
        
        try {
            switch ($action) {
                case 'upload':
                    // File upload processing
                    if (isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
                        $fileName = $_FILES['file']['name'];
                        $fileSize = $_FILES['file']['size'];
                        $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                        $uniqueFilename = uniqid() . '_' . $fileName;
                        $targetPath = $uploadDir . '/' . $uniqueFilename;
                        $isPremium = isset($_POST['is_premium']) ? 1 : 0;
                        
                        // Check if directory exists, if not create it
                        if (!is_dir(dirname($targetPath))) {
                            mkdir(dirname($targetPath), 0755, true);
                        }
                        
                        // Move uploaded file
                        if (move_uploaded_file($_FILES['file']['tmp_name'], $targetPath)) {
                            $relativeFilePath = 'public/files/' . date('Y-m-d') . '/' . $uniqueFilename;
                            
                            // Save to database
                            $stmt = $db->prepare("
                             INSERT INTO files (file_name, file_path, file_size, file_type, category_id, description, is_premium) 
                             VALUES (?, ?, ?, ?, ?, ?, ?)
                             ");
                            $stmt->execute([$fileName, $relativeFilePath, $fileSize, $fileType, !empty($_POST['category']) ? (int)$_POST['category'] : null,
                             $_POST['description'] ?? null, $isPremium]);
                            
                            $successMessage = "File berhasil diupload.";
                        } else {
                            $errorMessage = "Gagal mengupload file.";
                        }
                    } else {
                        $errorMessage = "Terjadi kesalahan: " . uploadErrorMessage($_FILES['file']['error']);
                    }
                    break;
                    
                case 'toggle_premium':
                    $stmt = $db->prepare("UPDATE files SET is_premium = 1 - is_premium WHERE id = ?");
                    $stmt->execute([$fileId]);
                    $successMessage = "Status premium file berhasil diubah.";
                    break;
                    
                case 'delete':
                    // Get file path first
                    $stmt = $db->prepare("SELECT file_path FROM files WHERE id = ?");
                    $stmt->execute([$fileId]);
                    $file = $stmt->fetch();
                    
                    if ($file) {
                        $filePath = __DIR__ . '/../' . $file['file_path'];
                        
                        // Delete file from storage if exists
                        if (file_exists($filePath)) {
                            unlink($filePath);
                        }
                        
                        // Delete from database
                        $stmt = $db->prepare("DELETE FROM files WHERE id = ?");
                        $stmt->execute([$fileId]);
                        
                        $successMessage = "File berhasil dihapus.";
                    } else {
                        $errorMessage = "File tidak ditemukan.";
                    }
                    break;
                    
                case 'delete_selected':
                    if (isset($_POST['selected_files']) && is_array($_POST['selected_files']) && !empty($_POST['selected_files'])) {
                        $selectedFiles = $_POST['selected_files'];
                        $placeholders = implode(',', array_fill(0, count($selectedFiles), '?'));
                        
                        // Get file paths first
                        $stmt = $db->prepare("SELECT id, file_path FROM files WHERE id IN ($placeholders)");
                        $stmt->execute($selectedFiles);
                        $files = $stmt->fetchAll();
                        
                        // Delete files from storage
                        foreach ($files as $file) {
                            $filePath = __DIR__ . '/../' . $file['file_path'];
                            if (file_exists($filePath)) {
                                unlink($filePath);
                            }
                        }
                        
                        // Delete from database
                        $stmt = $db->prepare("DELETE FROM files WHERE id IN ($placeholders)");
                        $stmt->execute($selectedFiles);
                        
                        $deletedCount = $stmt->rowCount();
                        $successMessage = "Berhasil menghapus $deletedCount file.";
                    } else {
                        $errorMessage = "Tidak ada file yang dipilih.";
                    }
                    break;
            }
        } catch (PDOException $e) {
            $errorMessage = "Database error: " . $e->getMessage();
        } catch (Exception $e) {
            $errorMessage = "Error: " . $e->getMessage();
        }
    }
}

if (!empty($_POST['tags'])) {
    $fileId = $db->lastInsertId();
    $tags = array_map('trim', explode(',', $_POST['tags']));
    
    foreach ($tags as $tagName) {
        if (empty($tagName)) continue;
        
        // Insert or get tag
        $stmt = $db->prepare("INSERT IGNORE INTO tags (name) VALUES (?)");
        $stmt->execute([$tagName]);
        
        $tagId = $db->lastInsertId();
        if (!$tagId) {
            $tagId = $db->query("SELECT id FROM tags WHERE name = " . $db->quote($tagName))->fetchColumn();
        }
        
        // Link tag to file
        $stmt = $db->prepare("INSERT IGNORE INTO file_tags (file_id, tag_id) VALUES (?, ?)");
        $stmt->execute([$fileId, $tagId]);
    }
}

// Filter settings
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$sort = isset($_GET['sort']) ? $_GET['sort'] : 'newest';

// Build query based on filter
$query = "SELECT f.id, f.file_name, f.file_path, f.file_size, f.file_type, f.created_at, f.download_count, f.is_premium, 
          c.name AS category_name,
          GROUP_CONCAT(t.name SEPARATOR ', ') AS tags
          FROM files f
          LEFT JOIN categories c ON f.category_id = c.id
          LEFT JOIN file_tags ft ON f.id = ft.file_id
          LEFT JOIN tags t ON ft.tag_id = t.id
          WHERE 1=1";

if ($filter === 'premium') {
    $query .= " AND is_premium = 1";
} elseif ($filter === 'public') {
    $query .= " AND is_premium = 0";
}

if (!empty($search)) {
    $query .= " AND (f.file_name LIKE :search 
                OR f.file_type LIKE :search 
                OR f.description LIKE :search 
                OR c.name LIKE :search 
                OR t.name LIKE :search)";
}

// Group by before adding ORDER BY
$query .= " GROUP BY f.id";

// Sorting
if ($sort === 'oldest') {
    $query .= " ORDER BY created_at ASC";
} elseif ($sort === 'name') {
    $query .= " ORDER BY file_name ASC";
} elseif ($sort === 'size') {
    $query .= " ORDER BY file_size DESC";
} elseif ($sort === 'downloads') {
    $query .= " ORDER BY download_count DESC";
} else {
    $query .= " ORDER BY created_at DESC"; // newest
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Get total count for pagination
$countQuery = "SELECT COUNT(*) FROM (
    SELECT f.id
    FROM files f
    LEFT JOIN categories c ON f.category_id = c.id
    LEFT JOIN file_tags ft ON f.id = ft.file_id
    LEFT JOIN tags t ON ft.tag_id = t.id
    WHERE 1=1";

if ($filter === 'premium') {
    $countQuery .= " AND is_premium = 1";
} elseif ($filter === 'public') {
    $countQuery .= " AND is_premium = 0";
}

if (!empty($search)) {
    $countQuery .= " AND (f.file_name LIKE :search 
                    OR f.file_type LIKE :search 
                    OR f.description LIKE :search 
                    OR c.name LIKE :search 
                    OR t.name LIKE :search)";
}

$countQuery .= " GROUP BY f.id
) AS sub";
$stmt = $db->prepare($countQuery);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->execute();
$totalFiles = $stmt->fetchColumn();
$totalPages = ceil($totalFiles / $perPage);

// Add LIMIT clause at the end
$query .= " LIMIT :offset, :perPage";
$stmt = $db->prepare($query);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
$stmt->execute();
$files = $stmt->fetchAll();

// Get file statistics
$stmt = $db->query("SELECT COUNT(*) as total FROM files");
$totalAllFiles = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM files WHERE is_premium = 1");
$totalPremiumFiles = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM files WHERE is_premium = 0");
$totalPublicFiles = $stmt->fetch()['total'];

$stmt = $db->query("SELECT SUM(download_count) as total FROM files");
$totalDownloads = $stmt->fetch()['total'] ?? 0;

// Helper function for upload errors
function uploadErrorMessage($error) {
    switch ($error) {
        case UPLOAD_ERR_INI_SIZE:
            return "File melebihi batas ukuran upload yang diizinkan.";
        case UPLOAD_ERR_FORM_SIZE:
            return "File melebihi batas ukuran form yang diizinkan.";
        case UPLOAD_ERR_PARTIAL:
            return "File hanya terupload sebagian.";
        case UPLOAD_ERR_NO_FILE:
            return "Tidak ada file yang diupload.";
        case UPLOAD_ERR_NO_TMP_DIR:
            return "Direktori temporary tidak ada.";
        case UPLOAD_ERR_CANT_WRITE:
            return "Gagal menulis file ke disk.";
        case UPLOAD_ERR_EXTENSION:
            return "Upload dihentikan oleh ekstensi.";
        default:
            return "Terjadi kesalahan yang tidak diketahui.";
    }
}

// Format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Page title
$pageTitle = "Kelola File";
?>

<?php include 'include/header.php'; ?>

<div class="container-fluid px-4">
    <h1 class="mt-4">Kelola File</h1>
    <ol class="breadcrumb mb-4">
        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
        <li class="breadcrumb-item active">Kelola File</li>
    </ol>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php echo $successMessage; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?php echo $errorMessage; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-lg-3 col-md-6">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h4 class="mb-0"><?php echo $totalAllFiles; ?></h4>
                            <div>Total File</div>
                        </div>
                        <div>
                            <i class="bi bi-file-earmark fs-1"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-3 col-md-6">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h4 class="mb-0"><?php echo $totalPublicFiles; ?></h4>
                            <div>File Publik</div>
                        </div>
                        <div>
                            <i class="bi bi-file-earmark-text fs-1"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-3 col-md-6">
            <div class="card bg-warning text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h4 class="mb-0"><?php echo $totalPremiumFiles; ?></h4>
                            <div>File Premium</div>
                        </div>
                        <div>
                            <i class="bi bi-file-earmark-lock fs-1"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-3 col-md-6">
            <div class="card bg-danger text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h4 class="mb-0"><?php echo $totalDownloads; ?></h4>
                            <div>Total Download</div>
                        </div>
                        <div>
                            <i class="bi bi-download fs-1"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Upload File dan Filter -->
    <div class="row mb-4">
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <i class="bi bi-upload me-1"></i> Upload File Baru
                </div>
                <div class="card-body">
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="upload">
                        
                        <div class="mb-3">
                            <label for="description" class="form-label">Deskripsi File</label>
                            <textarea class="form-control" id="description" name="description" rows="2"></textarea>
                        </div>
                        
                        <div class="mb-3">
                            <label for="category" class="form-label">Kategori</label>
                            <select class="form-select" id="category" name="category">
                            <option value="">-- Pilih Kategori --</option>
                              <?php 
                                 $categories = $db->query("SELECT id, name FROM categories ORDER BY name")->fetchAll();
                                 foreach ($categories as $cat): ?>
                            <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['name']) ?></option>
                                 <?php endforeach; ?>
                              </select>
                        </div>

                        <div class="mb-3">
                             <label for="tags" class="form-label">Tags (pisahkan dengan koma)</label>
                             <input type="text" class="form-control" id="tags" name="tags" placeholder="contoh: lua, script, bot">
                        </div>
                        
                        <div class="mb-3">
                            <label for="file" class="form-label">Pilih File</label>
                            <input type="file" class="form-control" id="file" name="file" required>
                            <div class="form-text">Ukuran maksimal file: <?php echo ini_get('upload_max_filesize'); ?></div>
                        </div>
                        
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="is_premium" name="is_premium">
                            <label class="form-check-label" for="is_premium">File Premium</label>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="bi bi-upload me-1"></i> Upload File
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header">
                    <i class="bi bi-funnel me-1"></i> Filter & Pencarian
                </div>
                <div class="card-body">
                    <form method="GET" class="row g-3">
                        <div class="col-md-5">
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-search"></i></span>
                                <input type="text" class="form-control" name="search" placeholder="Cari file..." value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <select class="form-select" name="filter">
                                <option value="all" <?php echo $filter === 'all' ? 'selected' : ''; ?>>Semua File</option>
                                <option value="public" <?php echo $filter === 'public' ? 'selected' : ''; ?>>File Publik</option>
                                <option value="premium" <?php echo $filter === 'premium' ? 'selected' : ''; ?>>File Premium</option>
                            </select>
                        </div>
                        
                        <div class="col-md-3">
                            <select class="form-select" name="sort">
                                <option value="newest" <?php echo $sort === 'newest' ? 'selected' : ''; ?>>Terbaru</option>
                                <option value="oldest" <?php echo $sort === 'oldest' ? 'selected' : ''; ?>>Terlama</option>
                                <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Nama</option>
                                <option value="size" <?php echo $sort === 'size' ? 'selected' : ''; ?>>Ukuran</option>
                                <option value="downloads" <?php echo $sort === 'downloads' ? 'selected' : ''; ?>>Jumlah Download</option>
                            </select>
                        </div>
                        
                        <div class="col-md-1">
                            <button type="submit" class="btn btn-primary w-100"><i class="bi bi-funnel"></i></button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Daftar File -->
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <div>
                <i class="bi bi-file-earmark-text me-1"></i> Daftar File
            </div>
            <div class="ms-auto">
                <?php if (!empty($files)): ?>
                <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#deleteSelectedModal">
                    <i class="bi bi-trash"></i> Hapus yang Dipilih
                </button>
                <?php endif; ?>
            </div>
        </div>
        <div class="card-body">
            <form id="fileForm" method="POST">
                <input type="hidden" name="action" id="formAction" value="">
                <input type="hidden" name="file_id" id="fileId" value="">
                
                <?php if (empty($files)): ?>
                    <div class="alert alert-info">
                        Belum ada file yang diupload.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th width="30">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="selectAll">
                                        </div>
                                    </th>
                                    <th>ID</th>
                                    <th>Nama File</th>
                                    <th>Tipe</th>
                                    <th>Ukuran</th>
                                    <th>Status</th>
                                    <th>Download</th>
                                    <th>Kategori</th>
                                    <th>Tags</th>
                                    <th>Tanggal</th>
                                    <th width="200">Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($files as $file): ?>
                                    <tr>
                                        <td>
                                            <div class="form-check">
                                                <input class="form-check-input file-checkbox" type="checkbox" name="selected_files[]" value="<?php echo $file['id']; ?>">
                                            </div>
                                        </td>
                                        <td><?php echo $file['id']; ?></td>
                                        <td>
                                            <a href="<?php echo '../' . $file['file_path']; ?>" target="_blank" title="Lihat file">
                                                <?php echo htmlspecialchars($file['file_name']); ?>
                                            </a>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?php echo $file['file_type']; ?></span>
                                        </td>
                                        <td><?php echo formatFileSize($file['file_size']); ?></td>
                                        <td>
                                            <?php if ($file['is_premium']): ?>
                                                <span class="badge bg-warning">Premium</span>
                                            <?php else: ?>
                                                <span class="badge bg-success">Publik</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo $file['download_count']; ?></td>
                                        <td><?= $file['category_name'] ?? '-' ?></td>
<td>
    <?php if (!empty($file['tags'])): ?>
        <?= implode(', ', array_map('htmlspecialchars', explode(',', $file['tags']))) ?>
    <?php else: ?>
        -
    <?php endif; ?>
</td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($file['created_at'])); ?></td>
                                        <td>
                                            <div class="d-flex gap-1">
                                                <button type="button" class="btn btn-sm btn-primary preview-button" 
                                                        data-filepath="<?php echo '../' . $file['file_path']; ?>" 
                                                        data-filename="<?php echo htmlspecialchars($file['file_name']); ?>"
                                                        data-filetype="<?php echo $file['file_type']; ?>">
                                                    <i class="bi bi-eye"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-warning toggle-premium-button" 
                                                        data-id="<?php echo $file['id']; ?>" 
                                                        data-is-premium="<?php echo $file['is_premium']; ?>">
                                                    <?php if ($file['is_premium']): ?>
                                                        <i class="bi bi-unlock"></i>
                                                    <?php else: ?>
                                                        <i class="bi bi-lock"></i>
                                                    <?php endif; ?>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-danger delete-button" 
                                                        data-id="<?php echo $file['id']; ?>" 
                                                        data-name="<?php echo htmlspecialchars($file['file_name']); ?>">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav>
                            <ul class="pagination justify-content-center">
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&filter=<?php echo $filter; ?>&search=<?php echo urlencode($search); ?>&sort=<?php echo $sort; ?>">
                                        <i class="bi bi-chevron-left"></i>
                                    </a>
                                </li>
                                
                                <?php
                                $startPage = max(1, $page - 2);
                                $endPage = min($totalPages, $page + 2);
                                
                                if ($startPage > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page=1&filter=' . $filter . '&search=' . urlencode($search) . '&sort=' . $sort . '">1</a></li>';
                                    if ($startPage > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                
                                for ($i = $startPage; $i <= $endPage; $i++) {
                                    $activeClass = $i == $page ? 'active' : '';
                                    echo '<li class="page-item ' . $activeClass . '"><a class="page-link" href="?page=' . $i . '&filter=' . $filter . '&search=' . urlencode($search) . '&sort=' . $sort . '">' . $i . '</a></li>';
                                }
                                
                                if ($endPage < $totalPages) {
                                    if ($endPage < $totalPages - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page=' . $totalPages . '&filter=' . $filter . '&search=' . urlencode($search) . '&sort=' . $sort . '">' . $totalPages . '</a></li>';
                                }
                                ?>
                                
                                <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&filter=<?php echo $filter; ?>&search=<?php echo urlencode($search); ?>&sort=<?php echo $sort; ?>">
                                        <i class="bi bi-chevron-right"></i>
                                    </a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </form>
        </div>
    </div>
</div>

<!-- Modal Konfirmasi Hapus -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Konfirmasi Hapus</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Apakah Anda yakin ingin menghapus file <strong id="deleteFileName"></strong>?</p>
                <p class="text-danger">Tindakan ini tidak dapat dibatalkan.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-danger" id="confirmDelete">Hapus</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal Konfirmasi Hapus yang Dipilih -->
<div class="modal fade" id="deleteSelectedModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Konfirmasi Hapus File</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Apakah Anda yakin ingin menghapus semua file yang dipilih?</p>
                <p class="text-danger">Tindakan ini tidak dapat dibatalkan.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteSelected">Hapus</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal Preview File -->
<div class="modal fade" id="previewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="previewFileName">Preview File</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="previewContent">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <a href="#" class="btn btn-primary" id="downloadPreview" target="_blank">Download</a>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>


<!-- Script untuk action buttons -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Select all checkbox
    const selectAll = document.getElementById('selectAll');
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.file-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
        });
    }
    
    // Delete button
    const deleteButtons = document.querySelectorAll('.delete-button');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
            const fileId = this.getAttribute('data-id');
            const fileName = this.getAttribute('data-name');
            
            document.getElementById('fileId').value = fileId;
            document.getElementById('formAction').value = 'delete';
            document.getElementById('deleteFileName').textContent = fileName;
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));
            deleteModal.show();
        });
    });
    
    // Confirm delete
    document.getElementById('confirmDelete').addEventListener('click', function() {
        document.getElementById('fileForm').submit();
    });
    
    // Delete selected
    document.getElementById('confirmDeleteSelected').addEventListener('click', function() {
        document.getElementById('formAction').value = 'delete_selected';
        document.getElementById('fileForm').submit();
    });
    
    // Toggle premium
    const togglePremiumButtons = document.querySelectorAll('.toggle-premium-button');
    togglePremiumButtons.forEach(button => {
        button.addEventListener('click', function() {
            const fileId = this.getAttribute('data-id');
            document.getElementById('fileId').value = fileId;
            document.getElementById('formAction').value = 'toggle_premium';
            document.getElementById('fileForm').submit();
        });
    });
    
    // Preview file
    const previewButtons = document.querySelectorAll('.preview-button');
    previewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const filePath = this.getAttribute('data-filepath');
            const fileName = this.getAttribute('data-filename');
            const fileType = this.getAttribute('data-filetype');
            
            document.getElementById('previewFileName').textContent = fileName;
            document.getElementById('downloadPreview').href = filePath;
            
            const previewContent = document.getElementById('previewContent');
            previewContent.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>';
            
            // Handle different file types
            if (fileType === 'pdf') {
                previewContent.innerHTML = `<embed src="${filePath}" type="application/pdf" width="100%" height="500px">`;
            } else if (['jpg', 'jpeg', 'png', 'gif', 'webp'].includes(fileType)) {
                previewContent.innerHTML = `<img src="${filePath}" class="img-fluid" alt="${fileName}">`;
            } else if (['mp4', 'webm', 'ogg'].includes(fileType)) {
                previewContent.innerHTML = `<video controls class="w-100"><source src="${filePath}" type="video/${fileType}">Your browser does not support the video tag.</video>`;
            } else if (['mp3', 'wav', 'ogg'].includes(fileType)) {
                previewContent.innerHTML = `<audio controls class="w-100"><source src="${filePath}" type="audio/${fileType}">Your browser does not support the audio tag.</audio>`;
            } else if (['txt', 'log', 'md'].includes(fileType)) {
                // Fetch and display text content
                fetch(filePath)
                    .then(response => response.text())
                    .then(data => {
                        previewContent.innerHTML = `<pre class="border p-3 bg-light" style="max-height: 500px; overflow-y: auto;">${data}</pre>`;
                    })
                    .catch(error => {
                        previewContent.innerHTML = `<div class="alert alert-danger">Error loading file: ${error.message}</div>`;
                    });
            } else if (['doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'zip', 'rar'].includes(fileType)) {
                previewContent.innerHTML = `
                    <div class="text-center">
                        <div class="mb-4">
                            <i class="bi bi-file-earmark-${getFileIcon(fileType)}" style="font-size: 5rem;"></i>
                        </div>
                        <p>Preview tidak tersedia untuk file ${fileType.toUpperCase()}.</p>
                        <a href="${filePath}" class="btn btn-primary" download>Download File</a>
                    </div>
                `;
            } else {
                previewContent.innerHTML = `
                    <div class="text-center">
                        <div class="mb-4">
                            <i class="bi bi-file-earmark" style="font-size: 5rem;"></i>
                        </div>
                        <p>Preview tidak tersedia untuk tipe file ini.</p>
                        <a href="${filePath}" class="btn btn-primary" download>Download File</a>
                    </div>
                `;
            }
            
            const previewModal = new bootstrap.Modal(document.getElementById('previewModal'));
            previewModal.show();
        });
    });
    
    // Helper function for file icons
    function getFileIcon(fileType) {
        switch(fileType) {
            case 'pdf': return 'pdf';
            case 'doc':
            case 'docx': return 'word';
            case 'xls':
            case 'xlsx': return 'excel';
            case 'ppt':
            case 'pptx': return 'powerpoint';
            case 'zip':
            case 'rar': return 'zip';
            case 'txt':
            case 'log': return 'text';
            case 'jpg':
            case 'jpeg':
            case 'png':
            case 'gif':
            case 'webp': return 'image';
            case 'mp4':
            case 'webm':
            case 'ogg': return 'play';
            case 'mp3':
            case 'wav': return 'music';
            default: return 'earmark';
        }
    }
    
    // Check if at least one checkbox is selected
    function checkSelectedFiles() {
        const checkboxes = document.querySelectorAll('.file-checkbox:checked');
        const deleteSelectedButton = document.querySelector('[data-bs-target="#deleteSelectedModal"]');
        if (deleteSelectedButton) {
            deleteSelectedButton.disabled = checkboxes.length === 0;
        }
    }
    
    // Add change event to all checkboxes
    const fileCheckboxes = document.querySelectorAll('.file-checkbox');
    fileCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', checkSelectedFiles);
    });
    
    // Initial check
    checkSelectedFiles();
    
    // Dismiss alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
});
</script>

<?php include 'include/footer.php'; ?>