<?php
require_once 'auth_check.php';

$pageTitle = "Kelola Kategori";

// Process actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        $categoryId = isset($_POST['category_id']) ? (int)$_POST['category_id'] : 0;
        
        try {
            switch ($action) {
                case 'add':
                    $name = trim($_POST['name']);
                    $description = trim($_POST['description']);
                    
                    if (empty($name)) {
                        $errorMessage = "Nama kategori tidak boleh kosong";
                        break;
                    }
                    
                    $stmt = $db->prepare("INSERT INTO categories (name, description) VALUES (?, ?)");
                    $stmt->execute([$name, $description]);
                    $successMessage = "Kategori berhasil ditambahkan";
                    break;
                    
                case 'edit':
                    $name = trim($_POST['name']);
                    $description = trim($_POST['description']);
                    
                    if (empty($name)) {
                        $errorMessage = "Nama kategori tidak boleh kosong";
                        break;
                    }
                    
                    $stmt = $db->prepare("UPDATE categories SET name = ?, description = ? WHERE id = ?");
                    $stmt->execute([$name, $description, $categoryId]);
                    $successMessage = "Kategori berhasil diperbarui";
                    break;
                    
                case 'delete':
                    // Check if category is used
                    $stmt = $db->prepare("SELECT COUNT(*) FROM files WHERE category_id = ?");
                    $stmt->execute([$categoryId]);
                    $usedCount = $stmt->fetchColumn();
                    
                    if ($usedCount > 0) {
                        $errorMessage = "Kategori tidak dapat dihapus karena masih digunakan oleh $usedCount file";
                        break;
                    }
                    
                    $stmt = $db->prepare("DELETE FROM categories WHERE id = ?");
                    $stmt->execute([$categoryId]);
                    $successMessage = "Kategori berhasil dihapus";
                    break;
            }
        } catch (PDOException $e) {
            if ($e->errorInfo[1] == 1062) {
                $errorMessage = "Nama kategori sudah ada";
            } else {
                $errorMessage = "Database error: " . $e->getMessage();
            }
        }
    }
}

// Get all categories
$categories = $db->query("
    SELECT c.*, COUNT(f.id) as file_count 
    FROM categories c
    LEFT JOIN files f ON c.id = f.category_id
    GROUP BY c.id
    ORDER BY c.name
")->fetchAll();

include 'include/header.php';
?>

<div class="container-fluid px-4">
    <h1 class="mt-4">Kelola Kategori</h1>
    <ol class="breadcrumb mb-4">
        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
        <li class="breadcrumb-item active">Kategori</li>
    </ol>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $successMessage ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $errorMessage ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-4">
            <div class="card mb-4">
                <div class="card-header">
                    <i class="bi bi-plus-circle me-1"></i> Tambah Kategori Baru
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="action" value="add">
                        
                        <div class="mb-3">
                            <label for="name" class="form-label">Nama Kategori*</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="description" class="form-label">Deskripsi</label>
                            <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="bi bi-save me-1"></i> Simpan Kategori
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-8">
            <div class="card mb-4">
                <div class="card-header">
                    <i class="bi bi-list-ul me-1"></i> Daftar Kategori
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th width="50">ID</th>
                                    <th>Nama</th>
                                    <th>Deskripsi</th>
                                    <th>Jumlah File</th>
                                    <th width="120">Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($categories)): ?>
                                    <tr>
                                        <td colspan="5" class="text-center">Belum ada kategori</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($categories as $category): ?>
                                        <tr>
                                            <td><?= $category['id'] ?></td>
                                            <td><?= htmlspecialchars($category['name']) ?></td>
                                            <td><?= htmlspecialchars($category['description']) ?></td>
                                            <td><?= $category['file_count'] ?></td>
                                            <td>
                                                <div class="d-flex gap-1">
                                                    <button type="button" class="btn btn-sm btn-warning edit-category" 
                                                            data-id="<?= $category['id'] ?>"
                                                            data-name="<?= htmlspecialchars($category['name']) ?>"
                                                            data-description="<?= htmlspecialchars($category['description']) ?>">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-sm btn-danger delete-category" 
                                                            data-id="<?= $category['id'] ?>"
                                                            data-name="<?= htmlspecialchars($category['name']) ?>">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Edit Kategori -->
<div class="modal fade" id="editCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Kategori</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="category_id" id="editCategoryId" value="">
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="editCategoryName" class="form-label">Nama Kategori*</label>
                        <input type="text" class="form-control" id="editCategoryName" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="editCategoryDescription" class="form-label">Deskripsi</label>
                        <textarea class="form-control" id="editCategoryDescription" name="description" rows="3"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-primary">Simpan Perubahan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Hapus Kategori -->
<div class="modal fade" id="deleteCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Konfirmasi Hapus</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="category_id" id="deleteCategoryId" value="">
                
                <div class="modal-body">
                    <p>Apakah Anda yakin ingin menghapus kategori <strong id="deleteCategoryName"></strong>?</p>
                    <p class="text-danger">Tindakan ini tidak dapat dibatalkan.</p>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-danger">Hapus</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Edit category modal
    const editButtons = document.querySelectorAll('.edit-category');
    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            document.getElementById('editCategoryId').value = this.getAttribute('data-id');
            document.getElementById('editCategoryName').value = this.getAttribute('data-name');
            document.getElementById('editCategoryDescription').value = this.getAttribute('data-description');
            
            const modal = new bootstrap.Modal(document.getElementById('editCategoryModal'));
            modal.show();
        });
    });
    
    // Delete category modal
    const deleteButtons = document.querySelectorAll('.delete-category');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
            document.getElementById('deleteCategoryId').value = this.getAttribute('data-id');
            document.getElementById('deleteCategoryName').textContent = this.getAttribute('data-name');
            
            const modal = new bootstrap.Modal(document.getElementById('deleteCategoryModal'));
            modal.show();
        });
    });
});
</script>

<?php include 'include/footer.php'; ?>